package ch.brx.ldifviewer;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import android.content.Context;
import android.util.Xml;


public class TemplateReader {

	private static final String ns = null;

	private static final String fieldNames[] = {"field0", "field1", "field2", "field3", "field4", 
			"field5", "field6", "field7", "field8" };
	private static final String headerNames[] = {null, null, "header2", "header3", "header4", "header5", 
			"header6", "header7", "header8" };
	
	private String templateDescription = null;
	private String templateName = null;
	private float leftColumnWidthPercent = 50.0f;
	private ArrayList<TemplateExpression> expressions = new ArrayList<TemplateExpression>();
	private TemplateNameFields nameFields = null;
	private String characterset = null;
	private ArrayList<String> searchFields = null;
	private ArrayList<String> phoneFormats = null;
	
	private String headerValues[] = {null, null, null, null, null, null, null, null, null};
	private boolean headerDynamicAttributes[] = {false, false, false, false, false, false, false, false, false};
	private String fieldTemplateRules[] = {null, null, null, null, null, null, null, null, null};
	
	private Context ctx;
	
	class TemplateExpression {
		String name;
		String type;
		String fields;
	}
	
	class TemplateNameFields {
		String firstName;
		String lastName;
		String callName;
	}


	TemplateReader(Context ctx, String name)
	{
		InputStream inStream;

		this.ctx = ctx;
		
		try {
			File file = new File(name);
			if(file.exists())
			{
				inStream = new FileInputStream(name);				
			}
			else
			{
				System.out.println("Use internal Asset: " + name);
				inStream = ctx.getAssets().open(name);
			}
		}
		catch(IOException ioe)
		{
			System.err.println("Cannot open: " + name + "(" + ioe + ")" );
			return;
		}
		try {
			XmlPullParser parser = Xml.newPullParser();
			parser.setFeature(XmlPullParser.FEATURE_PROCESS_NAMESPACES, false);
			parser.setInput(inStream, null);
			parser.nextTag();
			readTemplate(parser);
		}
		catch(Exception e)
		{
			System.err.println("XML parse error: " + e);
		}
		try
		{
			inStream.close();
		}
		catch(IOException ioe)
		{

		}
	}
	
	public String getTemplateName()
	{
		if(templateName == null)
		{
			return "";
		}
		else
		{
			return templateName;
		}
	}
	
	public String getTemplateDescription()
	{
		if(templateDescription == null)
		{
			return "";
		}
		else
		{
			return templateDescription;
		}
	}

	public String getFieldTemplate(int id)
	{
		if(id >= fieldTemplateRules.length || fieldTemplateRules[id] == null)
		{
			return "";
		}

		return replaceStringRes(fieldTemplateRules[id]);
	}

	public String getHeaderValue(int id)
	{
		if(id >= headerValues.length || headerValues[id] == null)
		{
			return "";
		}

		return replaceStringRes(headerValues[id]);
	}
	
	public List<TemplateExpression> getExpressions()
	{
		return expressions;
	}
	
	public TemplateNameFields getNameFields()
	{
		if(nameFields != null)
		{
			return nameFields;
		}
		else
		{
			TemplateNameFields nf = new TemplateNameFields();
			nf.firstName = "givenName";
			nf.lastName = "sn";
			nf.callName = "cn";
			return nf;
		}
	}
	
	public float getLeftColumnWidth()
	{
		return leftColumnWidthPercent;
	}

	public boolean isHeaderDynamic(int id)
	{
		if(id >= headerDynamicAttributes.length)
		{
			return false;
		}

		return headerDynamicAttributes[id];
	}
	
	public String getCharacterSet()
	{
		if(characterset == null)
		{
			return "UTF-8";
		}
		else
		{
			return characterset;
		}
	}
	
	public List<String> getSearchFields()
	{
		if(searchFields == null)
		{
			return new ArrayList<String>();
		}
		else
		{
			return searchFields;
		}
	}
	
	public List<String> getPhoneFormats()
	{
		if(phoneFormats == null)
		{
			return new ArrayList<String>();
		}
		else
		{
			return phoneFormats;
		}
	}

	private String replaceStringRes(String s)
	{
		int start_ix,end_ix;
		String packageName = ctx.getPackageName();

		do {
			start_ix = s.indexOf("@string/");
			if(start_ix >= 0)
			{
				for(end_ix=start_ix + 8; end_ix < s.length(); end_ix++)
				{
					if(!Character.isLetter(s.charAt(end_ix)))
					{
						break;
					}
				}
				
				String name = s.substring(start_ix,end_ix);

				int resId = ctx.getResources().getIdentifier(name, "string", packageName);
				if(resId > 0)
				{
					String string_res = ctx.getString(resId);

					s = s.replace(name, string_res);
				}
				else
				{
					break;
				}
			}

		} while(start_ix >= 0);

		return s;

	}

	private void readTemplate(XmlPullParser parser) throws XmlPullParserException, IOException 
	{
		parser.require(XmlPullParser.START_TAG, ns, "contact_template");
		templateName = parser.getAttributeValue(null, "name");
		String lstr = parser.getAttributeValue(null, "leftcolumn"); 
		leftColumnWidthPercent = Float.parseFloat(lstr);
		while (parser.next() != XmlPullParser.END_TAG) {
			if (parser.getEventType() != XmlPullParser.START_TAG) {
				continue;
			}
			String name = parser.getName();
			
			//System.out.println("XML tag: " + name);
			
			// Starts by looking for the entry tag
			if (name.equals("field")) {
				readField(parser);
			} else if (name.equals("expressions")) {
				readExpressions(parser);
			} else if (name.equals("searchfields")) {
				readSearchfields(parser);
			} else if (name.equals("phoneformat")) {
					readPhoneformat(parser);
			} else if (name.equals("header")) {
				readHeader(parser);
			} else if (name.equals("description")) {
				readDescription(parser);
			} else if (name.equals("name")) {
				readName(parser);
			} else if (name.equals("characterset")) {
				readCharacterset(parser);
			} else {
				skip(parser);
			}
		}  
	}
	
	private void readExpressions(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "expressions");

		while (parser.next() != XmlPullParser.END_TAG) {
			if (parser.getEventType() != XmlPullParser.START_TAG) {
				continue;
			}
			String name = parser.getName();
			if (name.equals("expression")) {
				readExpression(parser);
			} else {
				skip(parser);
			}
		}
		parser.require(XmlPullParser.END_TAG, ns, "expressions");
	}
	
	private void readExpression(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "expression");
		TemplateExpression expr = new TemplateExpression();
		expr.name = parser.getAttributeValue(null, "name");
		expr.type = parser.getAttributeValue(null, "type");
		expr.fields = readText(parser);
		expressions.add(expr);
		parser.require(XmlPullParser.END_TAG, ns, "expression");
	}
	
	private void readSearchfields(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "searchfields");
		
		searchFields = new ArrayList<String>();

		while (parser.next() != XmlPullParser.END_TAG) {
			if (parser.getEventType() != XmlPullParser.START_TAG) {
				continue;
			}
			String name = parser.getName();
			if (name.equals("field")) {
				readSearchfield(parser);
			} else {
				skip(parser);
			}
		}
		parser.require(XmlPullParser.END_TAG, ns, "searchfields");
	}
	
	private void readSearchfield(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "field");
		searchFields.add(readText(parser));
		parser.require(XmlPullParser.END_TAG, ns, "field");
	}
	
	private void readPhoneformat(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "phoneformat");
		
		phoneFormats = new ArrayList<String>();

		while (parser.next() != XmlPullParser.END_TAG) {
			if (parser.getEventType() != XmlPullParser.START_TAG) {
				continue;
			}
			String name = parser.getName();
			if (name.equals("number")) {
				readPhoneformatNumber(parser);
			} else {
				skip(parser);
			}
		}
		parser.require(XmlPullParser.END_TAG, ns, "phoneformat");
	}
	
	private void readPhoneformatNumber(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "number");
		phoneFormats.add(readText(parser));
		parser.require(XmlPullParser.END_TAG, ns, "number");
	}
	
	private void readName(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "name");
		this.nameFields = new TemplateNameFields();
		this.nameFields.firstName = parser.getAttributeValue(null, "first");
		this.nameFields.lastName = parser.getAttributeValue(null, "last");
		this.nameFields.callName = parser.getAttributeValue(null, "call");
		readText(parser);
		parser.require(XmlPullParser.END_TAG, ns, "name");
	}

	private void readField(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "field");

		String id = parser.getAttributeValue(null, "id");

		while (parser.next() != XmlPullParser.END_TAG) {
			if (parser.getEventType() != XmlPullParser.START_TAG) {
				continue;
			}
			String name = parser.getName();
			if (name.equals("line")) {
				readLine(parser, id);
			} else {
				skip(parser);
			}
		}
		parser.require(XmlPullParser.END_TAG, ns, "field");
	}

	private void readLine(XmlPullParser parser, String id) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "line");
		String line = readText(parser);

		int ix = findStr(fieldNames, id);
		if(ix >= 0)
		{
			if(fieldTemplateRules[ix] == null)
			{
				fieldTemplateRules[ix] = line;
			}
			else
			{
				fieldTemplateRules[ix] = fieldTemplateRules[ix] + '\n' + line;
			}
		}

		parser.require(XmlPullParser.END_TAG, ns, "line");
	}

	private void readHeader(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "header");
		String id = parser.getAttributeValue(null, "id");
		String dynamic = parser.getAttributeValue(null, "dynamic");
		String header = readText(parser);

		int ix = findStr(headerNames, id);
		if(ix >= 0)
		{
			headerValues[ix] = header;
			if(dynamic != null)
			{
				if(dynamic.equals("1"))
				{
					headerDynamicAttributes[ix] = true;
				}
				else if(dynamic.equals("0"))
				{
					headerDynamicAttributes[ix] = false;
				}
				else
				{
					System.err.println("XML attribute 'dynamic' is not 0 or 1: " + dynamic);
				}
			}
		}

		parser.require(XmlPullParser.END_TAG, ns, "header");
	}

	private void readDescription(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "description");
		templateDescription = readText(parser);
		parser.require(XmlPullParser.END_TAG, ns, "description");
	}
	
	private void readCharacterset(XmlPullParser parser) throws XmlPullParserException, IOException {
		parser.require(XmlPullParser.START_TAG, ns, "characterset");
		characterset = readText(parser);
		parser.require(XmlPullParser.END_TAG, ns, "characterset");
	}

	private String readText(XmlPullParser parser) throws IOException, XmlPullParserException {
		String result = "";
		if (parser.next() == XmlPullParser.TEXT) {
			result = parser.getText();
			parser.nextTag();
		}
		return result;
	}


	private int findStr(String[] arr, String s)
	{
		for(int ix = 0; ix < arr.length; ix++)
		{
			if(arr[ix] != null)
			{
				if(arr[ix].equals(s))
				{
					return ix;
				}
			}
		}

		return -1;
	}

	private void skip(XmlPullParser parser) throws XmlPullParserException, IOException {
		if (parser.getEventType() != XmlPullParser.START_TAG) {
			throw new IllegalStateException();
		}
		int depth = 1;
		while (depth != 0) {
			switch (parser.next()) {
			case XmlPullParser.END_TAG:
				depth--;
				break;
			case XmlPullParser.START_TAG:
				depth++;
				break;
			}
		}
	}

}
